/*==============================================================================
BINNED SCATTER BANK VALUES PLOT
==============================================================================
Purpose:
This script generates binned scatter plots showing bank values under different run hypotheses against the average uninsured share of domestic deposits. 
Input:
- $path_clean/analysis_ready_data.dta (Prepared bank-quarter panel dataset)
- Global variables: $fig_binned_scatter_dec, $fig_binned_scatter_feb (output filenames)
- Global variable: $periods_to_plot (which periods to generate)

Output:
- $path_figures/${fig_binned_scatter_dec}.pdf (Dec 2021 figure)
- $path_figures/${fig_binned_scatter_feb}.pdf (Feb 2023 figure)

Paper Correspondence:
- Main Analysis: Figure 8 Panel A (Dec 2021) and Panel B (Feb 2023)

Last updated: July 03, 2025
==============================================================================*/

display "--- Starting Binned Scatter Bank Values Plot ---"

**===============================================================================
* Generate Binned Scatter Plots
*===============================================================================
* Purpose: Create the binned scatter plots for each requested period.

* Parse the periods to plot from global variable
local periods_list $periods_to_plot

* Loop through the desired periods
foreach suffix in `periods_list' {
    * Set locals based on the current suffix
    if "`suffix'" == "dec2021" {
        loc title "Dec 2021"
        loc period_filter "dec2021" // Filter for dec2021 period
        loc xvar "uninsuredsh_domdep" // use current period uninsured share 
        loc output_file "${fig_binned_scatter_dec}"
    }
    else if "`suffix'" == "feb2023" {
        loc title "Feb 2023"
        loc period_filter "feb2023" // Filter for feb2023 period
        * Define the x-variable (average uninsured share for the current period)
        loc xvar "uninsuredsh_domdep_avg" // use average uninsured share
        loc output_file "${fig_binned_scatter_feb}"
    }

    * Define the y-variables (bank value components for the current period)
    * In long format, these are simply v_noDF, v_0, v_1 filtered by period
    loc yvars "v_noDF v_0 v_1"

    display "Generating Figure 8 for `title'"

    * Generate the binned scatter plot using options from draft.do
    binscatter `yvars' `xvar' if period == "`period_filter'", ///
        n(30) /* Number of bins (from draft.do) */ ///
        xtitle("Uninsured share") /* X-axis title */ ///
        ytitle("Bank value") /* Y-axis title */ ///
        yscale(r(-4 16)) /* Y-axis range */ ///
        ylabel(-4(4)16,grid) /* Y-axis labels and grid */ ///
        graphregion(color(white)) /* Graph region background color */ ///
        xscale(r(0 0.8)) /* X-axis scale (from draft.do 'all' sample) */ ///
        xlabel(0(.2).8, tlcolor(black) tposition(crossing)) /* X-axis labels (from draft.do 'all' sample) */ ///
        xsize(7) /* Graph width (from draft.do) */ ///
        mc(red%30 midblue%30 sand%30) /* Marker colors and transparency (from draft.do) */ ///
        msymbol(O T S) ///
        lcolor(red blue sand) /* Line colors (from draft.do) */ ///
        plotregion(lcolor(black)) /* Plot region border color */ ///
        yline(0, lc(black) lp(dash)) /* Add a dashed line at y=0 */ ///
        legend(order(1 "A {&minus} D" 2 "A {&minus} D + DF{subscript:I}" 3 "A {&minus} D + DF{subscript:I} + DF{subscript:U}") pos(6) cols(3) ring(0)) /* Legend customization (from draft.do) */ ///
        scheme(s1color) /* Plot scheme */

    graph export "${path_figures}/`output_file'.pdf", replace

    display "--- Plot for `title' completed ---"
    display "--- Output saved as: `output_file'.pdf ---"
}

display "--- Binned Scatter Bank Values Plot completed ---"
